<?php
function handleSearchHistory($conn, $method, $id, $subResource) {
    switch ($method) {
        case 'GET':
            if ($id === 'all') {
                getAllSearchHistory($conn);
            } elseif ($id) {
                getUserSearchHistory($conn, $id);
            } else {
                sendResponse(['error' => 'User ID required'], 400);
            }
            break;
        case 'POST':
            createSearchHistory($conn);
            break;
        case 'DELETE':
            if ($id && $subResource) {
                deleteSearchHistoryEntry($conn, $id, $subResource);
            } elseif ($id) {
                clearUserSearchHistory($conn, $id);
            } else {
                sendResponse(['error' => 'User ID required'], 400);
            }
            break;
        default:
            sendResponse(['error' => 'Method not allowed'], 405);
    }
}

function getUserSearchHistory($conn, $userId) {
    $limit = isset($_GET['limit']) ? intval($_GET['limit']) : 50;
    if ($limit > 1000) $limit = 1000;

    $sql = "SELECT * FROM search_history WHERE user_id = ? ORDER BY search_date DESC LIMIT $limit";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param('i', $userId);
    $stmt->execute();
    $result = $stmt->get_result();

    $history = [];
    while ($row = $result->fetch_assoc()) {
        $history[] = [
            'id' => $row['id'],
            'userId' => $row['user_id'],
            'username' => $row['username'],
            'searchType' => $row['search_type'],
            'searchParams' => json_decode($row['search_params'], true),
            'resultsCount' => $row['results_count'],
            'searchDate' => $row['search_date']
        ];
    }
    $stmt->close();
    sendResponse($history);
}

function getAllSearchHistory($conn) {
    checkAdmin($conn);

    $limit = isset($_GET['limit']) ? intval($_GET['limit']) : 100;
    if ($limit > 1000) $limit = 1000;

    $sql = "SELECT * FROM search_history ORDER BY search_date DESC LIMIT $limit";
    $result = $conn->query($sql);

    $history = [];
    while ($row = $result->fetch_assoc()) {
        $history[] = [
            'id' => $row['id'],
            'userId' => $row['user_id'],
            'username' => $row['username'],
            'searchType' => $row['search_type'],
            'searchParams' => json_decode($row['search_params'], true),
            'resultsCount' => $row['results_count'],
            'searchDate' => $row['search_date']
        ];
    }
    sendResponse($history);
}

function createSearchHistory($conn) {
    $data = getJsonInput();

    $userId = $data['userId'] ?? null;
    $username = $data['username'] ?? '';
    $searchType = $data['searchType'] ?? 'location';
    $searchParams = json_encode($data['searchParams'] ?? []);
    $resultsCount = $data['resultsCount'] ?? 0;

    if (!$userId) {
        sendResponse(['error' => 'User ID required'], 400);
    }

    $stmt = $conn->prepare('INSERT INTO search_history (user_id, username, search_type, search_params, results_count) VALUES (?, ?, ?, ?, ?)');
    $stmt->bind_param('isssi', $userId, $username, $searchType, $searchParams, $resultsCount);

    if ($stmt->execute()) {
        $stmt->close();
        sendResponse(['success' => true, 'id' => $conn->insert_id], 201);
    } else {
        sendResponse(['error' => 'Failed to save search history'], 500);
    }
}

function clearUserSearchHistory($conn, $userId) {
    $stmt = $conn->prepare('DELETE FROM search_history WHERE user_id = ?');
    $stmt->bind_param('i', $userId);

    if ($stmt->execute()) {
        $stmt->close();
        sendResponse(['success' => true]);
    } else {
        sendResponse(['error' => 'Failed to clear search history'], 500);
    }
}

function deleteSearchHistoryEntry($conn, $userId, $entryId) {
    $stmt = $conn->prepare('DELETE FROM search_history WHERE id = ? AND user_id = ?');
    $stmt->bind_param('ii', $entryId, $userId);

    if ($stmt->execute()) {
        $stmt->close();
        sendResponse(['success' => true]);
    } else {
        sendResponse(['error' => 'Failed to delete entry'], 500);
    }
}
?>
