<?php
function handleUsers($conn, $method, $id) {
    switch ($method) {
        case 'GET':
            if ($id) {
                getUser($conn, $id);
            } else {
                getUsers($conn);
            }
            break;
        case 'POST':
            createUser($conn);
            break;
        case 'PUT':
            updateUser($conn, $id);
            break;
        case 'DELETE':
            deleteUser($conn, $id);
            break;
        default:
            sendResponse(['error' => 'Method not allowed'], 405);
    }
}

function getUsers($conn) {
    checkAdmin($conn);

    $result = $conn->query('SELECT user_id, user_n, Name_of_user, group_id, active FROM land_pass_user ORDER BY user_n');
    $users = [];
    while ($row = $result->fetch_assoc()) {
        $users[] = [
            'id' => $row['user_id'],
            'username' => $row['user_n'],
            'fullName' => $row['Name_of_user'],
            'groupId' => $row['group_id'],
            'active' => (bool)$row['active']
        ];
    }
    sendResponse($users);
}

function getUser($conn, $id) {
    checkAdmin($conn);

    $stmt = $conn->prepare('SELECT user_id, user_n, Name_of_user, group_id, active FROM land_pass_user WHERE user_id = ?');
    $stmt->bind_param('i', $id);
    $stmt->execute();
    $result = $stmt->get_result();

    if ($result->num_rows === 0) {
        sendResponse(['error' => 'User not found'], 404);
    }

    $row = $result->fetch_assoc();
    $stmt->close();

    sendResponse([
        'id' => $row['user_id'],
        'username' => $row['user_n'],
        'fullName' => $row['Name_of_user'],
        'groupId' => $row['group_id'],
        'active' => (bool)$row['active']
    ]);
}

function createUser($conn) {
    checkAdmin($conn);

    $data = getJsonInput();
    $username = $data['username'] ?? '';
    $password = $data['password'] ?? '';
    $fullName = $data['fullName'] ?? '';
    $groupId = $data['groupId'] ?? 2;

    if (empty($username) || empty($password)) {
        sendResponse(['error' => 'Username and password are required'], 400);
    }

    // Check if username exists
    $checkStmt = $conn->prepare('SELECT user_id FROM land_pass_user WHERE user_n = ?');
    $checkStmt->bind_param('s', $username);
    $checkStmt->execute();
    if ($checkStmt->get_result()->num_rows > 0) {
        sendResponse(['error' => 'Username already exists'], 400);
    }
    $checkStmt->close();

    $stmt = $conn->prepare('INSERT INTO land_pass_user (user_n, pass_w, Name_of_user, group_id, active) VALUES (?, ?, ?, ?, 1)');
    $stmt->bind_param('sssi', $username, $password, $fullName, $groupId);

    if ($stmt->execute()) {
        $userId = $conn->insert_id;
        $stmt->close();
        sendResponse([
            'success' => true,
            'user' => [
                'id' => $userId,
                'username' => $username,
                'fullName' => $fullName,
                'groupId' => $groupId,
                'active' => true
            ]
        ], 201);
    } else {
        sendResponse(['error' => 'Failed to create user'], 500);
    }
}

function updateUser($conn, $id) {
    checkAdmin($conn);

    if (!$id) {
        sendResponse(['error' => 'User ID required'], 400);
    }

    $data = getJsonInput();
    $updates = [];
    $params = [];
    $types = '';

    if (isset($data['username'])) {
        $updates[] = 'user_n = ?';
        $params[] = $data['username'];
        $types .= 's';
    }
    if (isset($data['fullName'])) {
        $updates[] = 'Name_of_user = ?';
        $params[] = $data['fullName'];
        $types .= 's';
    }
    if (isset($data['password']) && !empty($data['password'])) {
        $updates[] = 'pass_w = ?';
        $params[] = $data['password'];
        $types .= 's';
    }
    if (isset($data['groupId'])) {
        $updates[] = 'group_id = ?';
        $params[] = $data['groupId'];
        $types .= 'i';
    }
    if (isset($data['active'])) {
        $updates[] = 'active = ?';
        $params[] = $data['active'] ? 1 : 0;
        $types .= 'i';
    }

    if (empty($updates)) {
        sendResponse(['error' => 'No fields to update'], 400);
    }

    $params[] = $id;
    $types .= 'i';

    $sql = 'UPDATE land_pass_user SET ' . implode(', ', $updates) . ' WHERE user_id = ?';
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);

    if ($stmt->execute()) {
        $stmt->close();
        sendResponse(['success' => true]);
    } else {
        sendResponse(['error' => 'Failed to update user'], 500);
    }
}

function deleteUser($conn, $id) {
    checkAdmin($conn);

    if (!$id) {
        sendResponse(['error' => 'User ID required'], 400);
    }

    $stmt = $conn->prepare('DELETE FROM land_pass_user WHERE user_id = ?');
    $stmt->bind_param('i', $id);

    if ($stmt->execute()) {
        $stmt->close();
        sendResponse(['success' => true]);
    } else {
        sendResponse(['error' => 'Failed to delete user'], 500);
    }
}

function handleGroups($conn) {
    $result = $conn->query('SELECT group_id as id, group_name as name FROM land_user_groups ORDER BY group_id');
    $groups = [];
    while ($row = $result->fetch_assoc()) {
        $groups[] = $row;
    }
    sendResponse($groups);
}
?>
